/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Andy Jefferson - coding standards
2004 Andy Jefferson - added query methods and comments
2004 Erik Bengtson - added joinKeysValuesTo
2004 Erik Bengtson - added joinKeysToGet
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Interface representation of the querying capabilities of a backing store for a map.
 * This is part of the JDOQL "expression" design that will be refactored out when the 
 * new query engine replaces it.
 **/
public interface MapStoreQueryable
{
    /**
	 * Create a subquery for the given query that joins a MapStore value table to the owner table.
	 * This subquery can subsequently be used in an EXISTS expression to determine whether a Map is empty or not. 
     * @param stmt The parent query statement that will use this as a subquery
     * @param ownerMapping the mapping for the owner
     * @param ownerTe Table Expression for the owner that the subquery joins to
     * @param mapRangeVar   The range variable for the "Map" table. 
     * @return  A subquery for the given query that joins a MapStore value table to the owner table. 
     */
    QueryExpression getExistsSubquery(QueryExpression stmt,
            JavaTypeMapping ownerMapping,
            LogicSetExpression ownerTe,
            DatastoreIdentifier mapRangeVar);

    /**
     * Create a subquery for the size of the map.
     * @param stmt The parent query statement that will use this as a subquery
     * @param ownerMapping the mapping for the owner
     * @param ownerTe Table Expression for the owner that the subquery joins to
     * @param mapRangeVar Range variable for this subquery main table
     * @return  A subquery for the size.
     */
    QueryExpression getSizeSubquery(QueryExpression stmt,
                            JavaTypeMapping ownerMapping,
                            LogicSetExpression ownerTe,
                            DatastoreIdentifier mapRangeVar);

    /**
     * Utility to create a join for keys and values to be used in ai
     * containsEntry() query.
     * @param stmt The Query Statement to apply the join
     * @param parentStmt the parent Query Statement. If there is no parent, <code>parentStmt</code> must be equals to <code>stmt</code> 
     * @param ownerMapping Mapping for the owner
     * @param te Table Expression for the owner
     * @param mapRangeVar  The SQL alias, or "range variable", to assign to the
     *                     expression or to the main table.
     * @param filteredKeyType The Class Type for the filtered key
     * @param filteredValueType The Class Type for the filtered value
     * @param keyExpr the expression to the key field                    
     * @param valExpr Table Expression for the value
     * @param keyRangeVar  The SQL alias, or "range variable", to assign to the
     *                     expression or to the key table.
     * @param valueRangeVar The SQL alias, or "range variable", to assign to the
     *                      expression or to the value table.
     * @return an array with 2 elements of QueryColumnList. The first element
     * contains the columns from the key mapping and the second element the
     * columns from the value mapping
     */
    ScalarExpression[] joinKeysValuesTo(
                                QueryExpression stmt,
                                QueryExpression parentStmt,
                                JavaTypeMapping ownerMapping,
                                LogicSetExpression te,
                                DatastoreIdentifier mapRangeVar,
                                Class filteredKeyType,
                                Class filteredValueType,
                                ScalarExpression keyExpr,
                                ScalarExpression valExpr,
                                DatastoreIdentifier keyRangeVar,
                                DatastoreIdentifier valueRangeVar);
    
    /**
     * Utility to create a join for keys to be used in a containsKey() query.
     * @param stmt The Query Statement 
     * @param parentStmt the parent Query Statement. If there is no parent, <code>parentStmt</code> must be equals to <code>stmt</code> 
     * @param ownerMapping Mapping for the owner
     * @param te Table Expression for the owner
     * @param mapRangeVar   The range variable for the "Map" table. 
     * @param filteredKeyType The Class Type for the filtered key
     * @param keyExpr the expression to the key field. if not provided, obtain the expression of the ID of the table where filteredKeyType is stored
     * @param keyRangeVar  The SQL alias, or "range variable", to assign to the
     *                     expression or to the key table.
     * @return QueryColumnList with the columns from the key mapping 
     **/
    ScalarExpression joinKeysTo(
                                QueryExpression stmt,
                                QueryExpression parentStmt,
                                JavaTypeMapping ownerMapping,
                                LogicSetExpression te,
                                DatastoreIdentifier mapRangeVar,
                                Class filteredKeyType,
                                ScalarExpression keyExpr,
                                DatastoreIdentifier keyRangeVar);

    /**
     * Used as part of the Querying of Maps where a containsValue() is used.
     * @param stmt The Query Statement 
     * @param parentStmt the parent Query Statement. If there is no parent, <code>parentStmt</code> must be equals to <code>stmt</code> 
     * @param ownerMapping Mapping for the owner
     * @param ownerTe Table Expression for the owner
     * @param mapRangeVar   The range variable for the "Map" table. 
     * @param filteredValueType The Class Type for the filtered value
     * @param valExpr the expression to the value field. if not provided, obtain the expression of the ID of the table where filteredValueType is stored
     * @param valueRangeVar The SQL alias, or "range variable", to assign to the
     *                      expression or to the value table.
     * @return QueryColumnList with the columns from the value mapping 
     **/
    ScalarExpression joinValuesTo(
                                QueryExpression stmt,
                                QueryExpression parentStmt,
                                JavaTypeMapping ownerMapping,
                                LogicSetExpression ownerTe,
                                DatastoreIdentifier mapRangeVar,
                                Class filteredValueType,
                                ScalarExpression valExpr,
                                DatastoreIdentifier valueRangeVar);
    
    /**
     * Used as part of the Querying of Maps where a get(Key) is used.
     * @param stmt The Query Statement to apply the join
     * @param parentStmt the parent Query Statement. If there is no parent, <code>parentStmt</code> must be equals to <code>stmt</code> 
     * @param ownerMapping Mapping for the owner
     * @param ownerTe Table Expression for the owner
     * @param mapRangeVar  The SQL alias, or "range variable", to assign to the
     *                     expression or to the main table.
     * @param filteredKeyType The Class Type for the filtered key
     * @param keyRangeVar  The SQL alias, or "range variable", to assign to the
     *                     expression or to the key table.
     * @param valueRangeVar The SQL alias, or "range variable", to assign to the
     *                      expression or to the value table.
     * @return an array with 2 elements of QueryColumnList. The first element
     * contains the columns from the key mapping and the second element the
     * columns from the value mapping
     */
    ScalarExpression[] joinKeysToGet(
                                QueryExpression stmt,
                                QueryExpression parentStmt,
                                JavaTypeMapping ownerMapping,
                                LogicSetExpression ownerTe,
                                DatastoreIdentifier mapRangeVar,
                                Class filteredKeyType,
                                DatastoreIdentifier keyRangeVar,
                                DatastoreIdentifier valueRangeVar);
}