/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.cache;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import org.datanucleus.util.StringUtils;

/**
 * An object that is stored in the Level2 Cache keyed by the identity of the persistable object.
 * Comprises a map of the field values keyed by the absolute field number in the class, the loaded fields array,
 * and the version of the object that is represented with these values.
 * Where the field is a relation field (PC, Map, Collection, array) we store the OID of any referenced persistable 
 * object. This is used when regenerating the object, and recreating its relations.
 */
public class CachedPC implements Serializable
{
    /** Class of the object being cached. */
    private Class cls;

    /** Values for the fields, keyed by the abs field number. Any relation fields store the id of the related object. */
    private Map<Integer, Object> fieldValues = null;

    /** Version of the cached object (if any) - Long, Timestamp etc. */
    private Object version;

    /** The loaded fields array */
    private boolean[] loadedFields;

    /**
     * Constructor.
     * @param cls The class of the object
     * @param loadedFields The loaded fields
     * @param vers The version (optional)
     */
    public CachedPC(Class cls, boolean[] loadedFields, Object vers)
    {
        this.cls = cls;
        this.loadedFields = new boolean[loadedFields.length];
        for (int i=0;i<loadedFields.length;i++)
        {
            this.loadedFields[i] = loadedFields[i];
        }
        this.version = vers;
    }

    public Class getObjectClass()
    {
        return cls;
    }

    public void setFieldValue(Integer fieldNumber, Object value)
    {
        if (fieldValues == null)
        {
            fieldValues = new HashMap<Integer, Object>();
        }
        fieldValues.put(fieldNumber, value);
    }

    public Object getFieldValue(Integer fieldNumber)
    {
        if (fieldValues == null)
        {
            return null;
        }
        return fieldValues.get(fieldNumber);
    }

    /**
     * Version accessor.
     * @return Version of cached object
     */
    public Object getVersion()
    {
        return version;
    }

    /**
     * Accessor for the loaded fields of this object.
     * Note : the fields here are mutable.
     * @return The loaded fields
     */
    public boolean[] getLoadedFields()
    {
        return loadedFields;
    }

    /**
     * Accessor for the field numbers that have values cached.
     * @return (Absolute) field numbers for cached fields
     */
    public Integer[] getFieldNumbers()
    {
        if (fieldValues == null)
        {
            return null;
        }
        return fieldValues.keySet().toArray(new Integer[fieldValues.size()]);
    }

    /**
     * Method to return a sting form of the cached object.
     * Returns something like "CachedPC : version=1 loadedFlags=[YY]"
     * @return The string form
     */
    public String toString()
    {
        return "CachedPC : version=" + version +
            " loadedFlags=" + StringUtils.booleanArrayToString(loadedFields);
    }
}