/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.math.BigInteger;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of an Integer literal.
 *
 * @version $Revision: 1.17 $
 **/
public class IntegerLiteral extends NumericExpression implements Literal
{
    private final Number value;

    /** Raw value that this literal represents. */
    Object rawValue;

    private StatementText stUsingParameter = new StatementText();

    private final boolean useParameter;
    
    /**
     * Creates a integer literal
     * @param qs the QueryExpression
     * @param mapping the mapping
     * @param value the integer value
     */    
    public IntegerLiteral(QueryExpression qs, JavaTypeMapping mapping, Number value)
    {
        super(qs);
        this.mapping = mapping;

        this.value = value;
        st.append(String.valueOf(value));
        stUsingParameter.appendParameter(mapping,value);
        this.useParameter = true;
    }

    /**
     * Creates a integer literal. This constructor should only be used when the <code>value</code>
     * will not change if the Query is run several times.
     * @param qs the QueryExpression
     * @param mapping the mapping
     * @param value the integer value
     * @param useParameter whether to use parameter or a literal in the expression when preparing the statement
     */    
    public IntegerLiteral(QueryExpression qs, JavaTypeMapping mapping, Number value, boolean useParameter)
    {
        super(qs);
        this.mapping = mapping;
        this.value = value;
        st.append(String.valueOf(value));
        stUsingParameter.appendParameter(mapping,value);
        this.useParameter = useParameter;
    }

    public Object getValue()
    {
        return value;
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, NumericExpression.class);

        if (expr instanceof IntegerLiteral)
        {
            return new BooleanLiteral(qs, mapping, new BigInteger(value.toString()).compareTo(new BigInteger(((IntegerLiteral)expr).value.toString())) == 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value.intValue()));
            return new BooleanExpression(expr, OP_EQ, literal);
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, NumericExpression.class);

        if (expr instanceof IntegerLiteral)
        {
            return new BooleanLiteral(qs, mapping, new BigInteger(value.toString()).compareTo(new BigInteger(((IntegerLiteral)expr).value.toString())) != 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value.intValue()));
            return new BooleanExpression(expr, OP_NOTEQ, literal);
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new BooleanLiteral(qs, mapping, new BigInteger(value.toString()).compareTo(new BigInteger(((IntegerLiteral)expr).value.toString())) < 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value.intValue()));
            return new BooleanExpression(literal, OP_LT, expr);
        }            
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new BooleanLiteral(qs, mapping, new BigInteger(value.toString()).compareTo(new BigInteger(((IntegerLiteral)expr).value.toString())) <= 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value.intValue()));
            return new BooleanExpression(literal, OP_LTEQ, expr);
        }            
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new BooleanLiteral(qs, mapping, new BigInteger(value.toString()).compareTo(new BigInteger(((IntegerLiteral)expr).value.toString())) > 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value.intValue()));
            return new BooleanExpression(literal, OP_GT, expr);
        }            
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new BooleanLiteral(qs, mapping, new BigInteger(value.toString()).compareTo(new BigInteger(((IntegerLiteral)expr).value.toString())) >= 0);
        }
        else if (expr instanceof CharacterExpression)
        {
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value.intValue()));
            return new BooleanExpression(literal, OP_GTEQ, expr);
        }            
        else
        {
            return super.gteq(expr);
        }
    }
    
    /**
     * If both operands are instances of IntegerLiteral, the operation results in BigInteger type. TODO fix this to follow JVM type conversions
     */
    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).add(new BigInteger(((IntegerLiteral)expr).value.toString())));
        }
        else if (expr instanceof CharacterLiteral)
        {
            int v = ((CharacterLiteral)expr).getValue().toString().charAt(0);
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).add(new BigInteger(""+v)));
        }        
        else
        {
            return super.add(expr);
        }
    }

    /**
     * If both operands are instances of IntegerLiteral, the operation results in BigInteger type. TODO fix this to follow JVM type conversions
     */
    public ScalarExpression sub(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).subtract(new BigInteger(((IntegerLiteral)expr).value.toString())));
        }
        else if (expr instanceof CharacterLiteral)
        {
            int v = ((CharacterLiteral)expr).getValue().toString().charAt(0);
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).subtract(new BigInteger(""+v)));
        }         
        else
        {
            return super.sub(expr);
        }
    }

    /**
     * If both operands are instances of IntegerLiteral, the operation results in BigInteger type. TODO fix this to follow JVM type conversions
     */
    public ScalarExpression mul(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).multiply(new BigInteger(((IntegerLiteral)expr).value.toString())));
        }
        else if (expr instanceof CharacterLiteral)
        {
            int v = ((CharacterLiteral)expr).getValue().toString().charAt(0);
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).multiply(new BigInteger(""+v)));
        }        
        else
        {
            return super.mul(expr);
        }
    }

    /**
     * If both operands are instances of IntegerLiteral, the operation results in BigInteger type. TODO fix this to follow JVM type conversions
     */
    public ScalarExpression div(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).divide(new BigInteger(((IntegerLiteral)expr).value.toString())));
        }
        else if (expr instanceof CharacterLiteral)
        {
            int v = ((CharacterLiteral)expr).getValue().toString().charAt(0);
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).divide(new BigInteger(""+v)));
        }        
        else
        {
            return super.div(expr);
        }
    }

    /**
     * If both operands are instances of IntegerLiteral, the operation results in BigInteger type. TODO fix this to follow JVM type conversions
     */
    public ScalarExpression mod(ScalarExpression expr)
    {
        if (expr instanceof IntegerLiteral)
        {
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).mod(new BigInteger(((IntegerLiteral)expr).value.toString())));
        }
        else if (expr instanceof CharacterLiteral)
        {
            int v = ((CharacterLiteral)expr).getValue().toString().charAt(0);
            return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).mod(new BigInteger(""+v)));
        }         
        else
        {
            return super.mod(expr);
        }
    }

    /**
     * Operation results in BigInteger type. TODO fix this to follow JVM type conversions
     */
    public ScalarExpression neg()
    {
        return new IntegerLiteral(qs, mapping, new BigInteger(value.toString()).negate());
    }
    
    public StatementText toStatementText(int mode)
    {
        if (mode == ScalarExpression.FILTER && useParameter)
        {
            return stUsingParameter;
        }
        return super.toStatementText(mode);
    }

    /**
     * Method to save a "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @param val The raw value
     */
    public void setRawValue(Object val)
    {
        this.rawValue = val;
    }

    /**
     * Accessor for the "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @return The raw value
     */
    public Object getRawValue()
    {
        return rawValue;
    }
}