/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.vault.inmemory;

import static java.util.concurrent.CompletableFuture.runAsync;
import static java.util.concurrent.CompletableFuture.supplyAsync;

import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import org.apache.ignite.internal.util.Cursor;
import org.apache.ignite.internal.vault.VaultEntry;
import org.apache.ignite.internal.vault.VaultService;
import org.apache.ignite.lang.ByteArray;
import org.jetbrains.annotations.Nullable;

/**
 * Simple in-memory representation of the Vault Service.
 */
public class InMemoryVaultService implements VaultService {
    /** Map to store values. */
    private final NavigableMap<ByteArray, byte[]> storage = new TreeMap<>();

    /** Mutex. */
    private final Object mux = new Object();

    @Override
    public void start() {
        // No-op.
    }

    /** {@inheritDoc} */
    @Override
    public void close() {
        // No-op.
    }

    /** {@inheritDoc} */
    @Override
    public CompletableFuture<VaultEntry> get(ByteArray key) {
        return supplyAsync(() -> {
            synchronized (mux) {
                byte[] value = storage.get(key);

                return value == null ? null : new VaultEntry(key, storage.get(key));
            }
        });
    }

    /** {@inheritDoc} */
    @Override
    public CompletableFuture<Void> put(ByteArray key, byte @Nullable [] val) {
        return runAsync(() -> {
            synchronized (mux) {
                storage.put(key, val);
            }
        });
    }

    /** {@inheritDoc} */
    @Override
    public CompletableFuture<Void> remove(ByteArray key) {
        return runAsync(() -> {
            synchronized (mux) {
                storage.remove(key);
            }
        });
    }

    /** {@inheritDoc} */
    @Override
    public Cursor<VaultEntry> range(ByteArray fromKey, ByteArray toKey) {
        Iterator<VaultEntry> it;

        if (fromKey.compareTo(toKey) >= 0) {
            it = Collections.emptyIterator();
        } else {
            synchronized (mux) {
                it = storage.subMap(fromKey, toKey).entrySet().stream()
                        .map(e -> new VaultEntry(new ByteArray(e.getKey()), e.getValue()))
                        .iterator();
            }
        }

        return Cursor.fromIterator(it);
    }

    /** {@inheritDoc} */
    @Override
    public CompletableFuture<Void> putAll(Map<ByteArray, byte[]> vals) {
        return runAsync(() -> {
            synchronized (mux) {
                for (var entry : vals.entrySet()) {
                    if (entry.getValue() == null) {
                        storage.remove(entry.getKey());
                    } else {
                        storage.put(entry.getKey(), entry.getValue());
                    }
                }
            }
        });
    }
}
