/*
 * SessionManager.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1998-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, rinaldo, suhler.
 *
 * Version:  2.2
 * Created by suhler on 98/09/14
 * Last modified by suhler on 04/04/05 14:45:12
 */

package sunlabs.brazil.session;

import java.util.Hashtable;

/**
 * The <code>SessionManager</code> associates an object with a Session ID
 * to give Handlers the ability to maintain state that lasts for the
 * duration of a session instead of just for the duration of a request.
 * <p>
 * The <code>SessionManager</code> operates as a bag of globally
 * accessible resources.  Existing subclasses of the
 * <code>SessionManager</code> also provide persistence, that is, a way to
 * recover these resources even if the server process is terminated and
 * later restarted, to get back to the state things were in.
 * <p>
 * A session manager is like a Dictionary only with fewer guarantees.
 * Enumeration is not possible, and a "get" might return null even if there was 
 * a previous "put", so users should be prepared to handle that case.
 * <p>
 * Unlike a typical dictionary, a session manager uses two keys to identify
 * to identify each resource.  The first key, by convention, represents
 * a client session id.  The second (or resource) key is chosen to identify the
 * resource within a session.
 * <p>
 * Care should be used when choosing resource keys, as they are global
 * to a JVM, which may have several unrelated Brazil servers running at
 * once.  Sharing session manager resources between servers can cause
 * unexpected behavior if done unintentionally.
 * <p>
 * Existing session manager implementations arrange for session resources 
 * that are <code>Java Properties</code> to be saved across restarts
 * of the JVM, and should be used when practical.
 *
 * @author	Stephen Uhler (stephen.uhler@sun.com)
 * @author	Colin Stevens (colin.stevens@sun.com)
 * @version	2.2, 04/04/05
 */

public class SessionManager {
    private static SessionManager sm = new SessionManager();

    /**
     * Installs the given <code>SessionManager</code> object as the
     * default session manager to be invoked when <code>getSession</code>
     * is called.
     *
     * @param	mgr
     *		The <code>SessionManager</code> object.
     */

    public static void
    setSessionManager(SessionManager mgr) {
	sm = mgr;
    }

    /**
     * Returns the object associated with the given Session ID.  Passing in
     * the same (hash-key equivalent) Session ID will return the same object.
     * This convenience method reflects common usage.
     *
     * @param	session
     *		The Session ID for the persistent session information.  If
     *		the session does not exist, a new one is created.
     *
     * @param	ident
     *		An arbitray identifier used to determine which object
     *		(associated with the given <code>session</code>) the caller
     *		wants.
     *
     * @param	type
     *		The Class of the object to create.  If the given
     *		<code>session</code> and <code>ident</code> did not specify
     *		an existing object, a new one is created by calling
     *		<code>newInstance</code> based on the <code>type</code>.
     *		If <code>null</code>, then this method returns
     *		<code>null</code> if the object didn't exist, instead of
     *		allocating a new object.
     *
     * @return	an object of type <code>type</code>, or null if
     *		the object doesn't exist and <code>type</code> is null.
     */

    public static Object
    getSession(Object session, Object ident, Class type) {
        Object obj = sm.getObj(session, ident);
        if (type != null && obj == null) {
	    try {
		obj = type.newInstance();
	    } catch (Exception e) {
		throw new IllegalArgumentException(type.getName() +
			": " + e.getClass().getName());
	    }
	    sm.putObj(session, ident, obj);
	}
	return obj;
    }

    /*
     * Allow access to std get/put/remove methods of the installed
     * Session Manager.
     */

    /**
     * get an object from the session manager.
     * This static method will dispatch to the currently installed
     * <code>SessionManager</code> instance.
     */

    public static Object
    get(Object session, Object ident) {
	return sm.getObj(session, ident);
    }

    /**
     * put an object into the session manager.
     * This static method will dispatch to the currently installed
     * <code>SessionManager</code> instance.
     */

    public static void
    put(Object session, Object ident, Object data) {
	sm.putObj(session, ident, data);
    }

    /**
     * Remove an object from the session manager.
     * This static method will dispatch to the currently installed
     * <code>SessionManager</code> instance.
     */

    public static void
    remove(Object session, Object ident) {
	sm.removeObj(session, ident);
    }

    /* Default Implementation */

    /**
     * NOTE: The previous implementation breaks for java > 1.1.  Use
     *     this one instead.
     * A <code>Hashtable</code> used when mapping Session IDs to objects.
     * <p>
     * the key for the hashtable is derived from the "session" and "ident"
     * objects.
     */

    protected Hashtable sessions = new Hashtable();

    /**
     * Returns the object associated with the given Session ID and ident.
     */

    protected Object
    getObj(Object session, Object ident) {
	String key = makeKey(session, ident);
	return sessions.get(key);
    }

    /**
     * Associates an object with a session id and ident.
     * "value" may not be null.
     */

    protected void
    putObj(Object session, Object ident, Object value) {
	String key = makeKey(session, ident);
	sessions.put(key, value);
    }

    /**
     * Removes the object associated with the given Session ID and ident.
     */

    protected void
    removeObj(Object session, Object ident) {
	String key = makeKey(session, ident);
	sessions.remove(key);
    }

    /**
     * Invent a single key from the 2 separate ones
     */

    protected String
    makeKey(Object session, Object ident) {
	if (ident == null) {
	    ident = "null";
        }
	if (session == null) {
	    session = this.getClass();
	}
	if (session instanceof String && ident instanceof String) {
	    return session + "-" + ident;
	} else {
	    return "" + session.hashCode() + "-" + ident.hashCode();
	}
    }
}
